<?php

namespace App\Models;

use CodeIgniter\Model;

class InmuebleModel extends Model
{
    protected $table = 'inmueble';
    protected $primaryKey = 'ID';
    protected $useAutoIncrement = false;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    protected $allowedFields = [
        'precioDolar', 'precioSoles', 'tipo', 'estado', 'departamento', 'provincia', 'distrito',
        'direccion', 'direccionDetallada', 'habitaciones', 'banios', 'garages', 'areaTerreno',
        'areaConstruccion', 'pisos', 'anioConstruccion', 'descripcion', 'coorX', 'coorY',
        'oferta', 'destacado', 'vendido', 'visitas', 'email', 'fechaRegistro', 'habilitado', 'vencimiento'
    ];

    // Dates
    protected $useTimestamps = false;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    protected $deletedField = 'deleted_at';

    // Validation
    protected $validationRules = [];
    protected $validationMessages = [];
    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert = [];
    protected $afterInsert = [];
    protected $beforeUpdate = [];
    protected $afterUpdate = [];
    protected $beforeFind = [];
    protected $afterFind = [];
    protected $beforeDelete = [];
    protected $afterDelete = [];

    /**
     * Obtener inmuebles habilitados con paginación
     */
    public function getInmueblesHabilitados($limit = 10, $offset = 0)
    {
        return $this->select('inmueble.*, 
                              departamento.Nombre as dep, 
                              provincia.Nombre as pro, 
                              distrito.Nombre as dis,
                              tipoinmueble.tipoInmueble,
                              estado.estadoInmueble')
                    ->join('departamento', 'inmueble.departamento = departamento.IdDepartamento', 'left')
                    ->join('provincia', 'inmueble.provincia = provincia.IdProvincia', 'left')
                    ->join('distrito', 'inmueble.distrito = distrito.IdDistrito', 'left')
                    ->join('tipoinmueble', 'inmueble.tipo = tipoinmueble.idTipoInmueble', 'left')
                    ->join('estado', 'inmueble.estado = estado.idEstadoInmueble', 'left')
                    ->where('inmueble.habilitado', 1)
                    ->where('DATEDIFF(CURDATE(), inmueble.fechaRegistro) <= inmueble.vencimiento')
                    ->orderBy('inmueble.destacado', 'DESC')
                    ->orderBy('inmueble.ID', 'DESC')
                    ->limit($limit, $offset)
                    ->findAll();
    }

    /**
     * Contar total de inmuebles habilitados
     */
    public function countInmueblesHabilitados()
    {
        return $this->where('habilitado', 1)
                    ->where('DATEDIFF(CURDATE(), fechaRegistro) <= vencimiento')
                    ->countAllResults();
    }

    /**
     * Obtener inmueble por ID con datos relacionados
     */
    public function getInmuebleById($id)
    {
        return $this->select('inmueble.*, 
                              departamento.Nombre as dep, 
                              departamento.IdDepartamento as IDEP,
                              provincia.Nombre as pro, 
                              distrito.Nombre as dis,
                              tipoinmueble.tipoInmueble,
                              estado.estadoInmueble')
                    ->join('departamento', 'inmueble.departamento = departamento.IdDepartamento', 'left')
                    ->join('provincia', 'inmueble.provincia = provincia.IdProvincia', 'left')
                    ->join('distrito', 'inmueble.distrito = distrito.IdDistrito', 'left')
                    ->join('tipoinmueble', 'inmueble.tipo = tipoinmueble.idTipoInmueble', 'left')
                    ->join('estado', 'inmueble.estado = estado.idEstadoInmueble', 'left')
                    ->where('inmueble.ID', $id)
                    ->first();
    }

    /**
     * Obtener departamento por código
     */
    public function getDepartamento($codigo)
    {
        $db = \Config\Database::connect();
        $query = $db->table('departamento')
                    ->select('Nombre')
                    ->where('IdDepartamento', $codigo)
                    ->get();
        
        $result = $query->getRow();
        return $result ? $result->Nombre : '';
    }

    /**
     * Obtener provincia por código
     */
    public function getProvincia($codigo)
    {
        $db = \Config\Database::connect();
        $query = $db->table('provincia')
                    ->select('Nombre')
                    ->where('IdProvincia', $codigo)
                    ->get();
        
        $result = $query->getRow();
        return $result ? $result->Nombre : '';
    }

    /**
     * Obtener distrito por código
     */
    public function getDistrito($codigo)
    {
        $db = \Config\Database::connect();
        $query = $db->table('distrito')
                    ->select('Nombre')
                    ->where('IdDistrito', $codigo)
                    ->get();
        
        $result = $query->getRow();
        return $result ? $result->Nombre : '';
    }

    /**
     * Obtener tipo de inmueble por código
     */
    public function getTipoInmueble($codigo)
    {
        $db = \Config\Database::connect();
        $query = $db->table('tipoinmueble')
                    ->select('tipoInmueble')
                    ->where('idTipoInmueble', $codigo)
                    ->get();
        
        $result = $query->getRow();
        return $result ? $result->tipoInmueble : '';
    }

    /**
     * Obtener estado de inmueble por código
     */
    public function getEstadoInmueble($codigo)
    {
        $db = \Config\Database::connect();
        $query = $db->table('estado')
                    ->select('estadoInmueble')
                    ->where('idEstadoInmueble', $codigo)
                    ->get();
        
        $result = $query->getRow();
        return $result ? $result->estadoInmueble : '';
    }
}
